/*	DOS Stamp helper functions
	(C)1999 Bagotronix Inc.
	Author: Ivan Baggett

	Written in Borland C v3.0.
*/

#include "stdinc.h"
#include "dosstamp.h"


/*	Bit masks for PIO configuration.  A 1 in a bit position means that bit
	can be modified.  A 0 means it cannot.

	PIOs 2-9,14,15,22-26,29 are not allowed to be changed.  They are used
	by critical functions.
*/
UWORD piomodes[32] = {
	0x0001,	/* PIO0 */
	0x0002,	/* PIO1 */
	0x0000,	/* PIO2, not available */
	0x0000,	/* PIO3, not available */
	0x0000,	/* PIO4, not available */
	0x0000,	/* PIO5, not available */
	0x0000,	/* PIO6, not available */
	0x0000,	/* PIO7, not available */
	0x0000,	/* PIO8, not available */
	0x0000,	/* PIO9, not available */
	0x0400,	/* PIO10 */
	0x0800,	/* PIO11 */
	0x1000,	/* PIO12 */
	0x2000,	/* PIO13 */
	0x0000,	/* PIO14, not available */
	0x0000,	/* PIO15, not available */

	0x0001,	/* PIO16 */
	0x0002,	/* PIO17 */
	0x0004,	/* PIO18 */
	0x0008,	/* PIO19 */
	0x0010,	/* PIO20 */
	0x0020,	/* PIO21 */
	0x0000,	/* PIO22, not available */
	0x0000,	/* PIO23, not available */
	0x0000,	/* PIO24, not available */
	0x0000,	/* PIO25, not available */
	0x0000,	/* PIO26, not available */
	0x0800,	/* PIO27 */
	0x1000,	/* PIO28 */
	0x0000,	/* PIO29, not available */
	0x4000,	/* PIO30 */
	0x8000	/* PIO31 */
};



/****************************************************************************
PIOconfig():  configures a PIO pin

	arguments:
		1) PIO number (0 - 31)
		2) PIO configuration (0 - 3)
		3) PIO pin initial value (0 = low, 1 = high)
	returns:  nothing
	resources used:  the PIO pin specified, PIO control registers
	comments:
		Interrupts are not disabled during this function.  You may need to
		disable them before calling this function and reenable them
		afterwards.
	verified:
****************************************************************************/
void PIOconfig (UWORD pionum, UWORD piocfg, BOOLEAN initval) {
	UWORD bitmask;
	UWORD temp;
	UWORD ofs;

	if (pionum < sizeof piomodes / sizeof (UWORD)) { 	// check bounds
		ofs = pionum / 16;		// select the PIO register set (0 or 1)
		ofs = ofs ? 6 : 0;	// convert to an offset
		bitmask = piomodes[pionum];	// get PIO bit mask

		// write the PIO data
		temp = inport (PIO0_DATA + ofs);
		outport (PIO0_DATA + ofs,
			(temp & ~bitmask) | (initval ? bitmask : 0));

		// write the PIO mode
		temp = inport (PIO0_MODE + ofs);
		outport (PIO0_MODE + ofs,
			(temp & ~bitmask) | ((piocfg & 0x0002) ? bitmask : 0));

		// write the PIO direction
		temp = inport (PIO0_DIR + ofs);
		outport (PIO0_DIR + ofs,
			(temp & ~bitmask) | ((piocfg & 0x0001) ? bitmask : 0));
	}
}


/****************************************************************************
IRQinit():  programs IRQ channel for use

	arguments:  the vector #, ptr to interrupt handler function, priority,
		trigger spec
	returns:  nothing
	resources used:
	comments:  DOES NOT UNMASK THE INTERRUPT
		vector #:  0 - 255
		priority:  0=highest, 7 = lowest
		trigger:   0=edge, non-zero=level (does not apply to internal
			interrupts, but argument must be passed anyway)
	verified:
****************************************************************************/
void IRQinit (UWORD vecnum, void interrupt (*vec)(), UWORD priority,
	BOOLEAN trigger) {
	UWORD temp;

#if 1
	disable ();
	setvect (vecnum, vec);
	enable ();
	switch (vecnum) {
		case INT0_VEC:
		case INT1_VEC:
		case INT2_VEC:
		case INT3_VEC:
			/* calculate interrupt control reg offset */
			temp = (vecnum - INT0_VEC) * 2;
			/* program trigger and priority and leave masked */
			outport (INT0_CON + temp, (trigger ? 0x0010 : 0)
				| ((UWORD)priority & 0x0007) | 0x0008);
			break;
		default:
			break;
	}
#endif
}


void unmask (UWORD intnum) {
#if 1
	outport (IMASK, inport (IMASK) & ~(1 << (intnum - 8)));
#endif
}


void mask (UWORD intnum) {
#if 1
/*	MASK usage:  MASK(x); where x is the interrupt number (8-15) */
	outport (IMASK, inport (IMASK) | (1 << (intnum - 8)));
#endif
}


/*	Convert unsigned byte to bcd byte

	arguments:  unsigned byte (0 - 99, or 0 - 63h)
	returns:  bcd equivalent in an unsigned byte
	comments:  for use by RTC operations
		Gives invalid results for out of range argument.
*/
UBYTE ubtobcd (UBYTE ub) {
	UWORD u;

	u = (ub / 10) << 4;
	u += ub % 10;
	return ((UBYTE)u);
}

/*	Convert bcd byte to unsigned binary byte

	arguments:  bcd byte (0 - 99h)
	returns:  unsigned binary byte equivalent of bcd
	comments:  for use by RTC operations
		Gives invalid results for out of range argument.
*/
UBYTE bcdtoub (UBYTE bcd) {
	UWORD u;

	u = (bcd >> 4) * 10;
	u += bcd & 0x0f;
	return ((UBYTE)u);
}

