/****************************************************************************
	Bagotronix
	1019 Crossing Brook Way
	Tallahassee, FL 32311

	(C) Copyright 1996 Bagotronix
	All rights reserved

	Filename:  		csfc8250.c
	Programmer(s):	Ivan Baggett
	Description:	Sunflower C(tm) library
					Chickadee(tm) 8250 serial communication functions
	comments:		for 8250 and compatibles
****************************************************************************/

#include <dos.h>

#include "stdinc.h"
#include "chickade.h"
#include "csfc8250.h"


/****************************************************************************
_8250_disable():  disables 8250 interrupt pin

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  does not mask the CPU interrupt
****************************************************************************/
void _8250_disable (IOPARMS *parms) {
	disable ();
	outportb (_MCR, inportb (_MCR) & ~0x08);	/* disable int pin */
	enable ();
}


/****************************************************************************
_8250_enable():  enables 8250 interrupt pin

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  does not unmask the CPU interrupt
****************************************************************************/
void _8250_enable (IOPARMS *parms) {
	disable ();
	outportb (_MCR, inportb (_MCR) | 0x08);	/* enable int pin */
	enable ();
}


/****************************************************************************
_8250_setbaud():  sets the baud rate of the 8250

	arguments:  ptr to comm parameters, baud divisor
	returns:  nothing
	resources used:  the comm port
	comments:  the value to use for baud divisor is calculated by this
		formula:  bauddiv = 115,200 / desired baud rate
****************************************************************************/
void _8250_setbaud (IOPARMS *parms, UWORD bauddiv) {
	disable ();
	outportb (_LCR, inportb (_LCR) | 0x80);		/* divisor latch access */
	outportb (_DLL, bauddiv);					/* divisor low byte */
	outportb (_DLM, bauddiv >> 8);				/* divisor high byte */
	outportb (_LCR, inportb (_LCR) & ~0x80);	/* rcvr buffer access */
	enable ();
}


/****************************************************************************
_8250_setparity():  set the parity of the 8250

	arguments:  ptr to comm parameters, the parity value
	returns:  nothing
	resources used:  the comm port
	comments:  parity values (0-2) 0 = none, 1 = odd, 2 = even
****************************************************************************/
void _8250_setparity (IOPARMS *parms, UBYTE parity) {
	UBYTE temp;

	temp = parity ? (((parity - 1) << 4) | 0x08) : 0;
	disable ();
	outportb (_LCR, (inportb (_LCR) & ~0x18) | temp);	/* configure UART */
	enable ();
}


/****************************************************************************
_8250_setdatabits():  sets data format of the 8250

	arguments:  ptr to comm parameters, # of data bits
	returns:  nothing
	resources used:  the comm port
	comments:  data bits (5-8)
****************************************************************************/
void _8250_setdatabits (IOPARMS *parms, UBYTE databits) {
	UBYTE temp;

	temp = databits - 5;
	disable ();
	outportb (_LCR, (inportb (_LCR) & ~0x03) | temp);	/* configure UART */
	enable ();
}


/****************************************************************************
_8250_setstopbits():  set the # of stop bits for the 8250

	arguments:  ptr to comm parameters, # of stop bits
	returns:  nothing
	resources used:  the comm port
	comments:  stop bits (1-2)
****************************************************************************/
void _8250_setstopbits (IOPARMS *parms, UBYTE stopbits) {
	UBYTE temp;

	temp = (stopbits - 1) << 2;
	disable ();
	outportb (_LCR, (inportb (_LCR) & ~0x04) | temp);	/* configure UART */
	enable ();
}


/****************************************************************************
_8250_setinterrupts():  sets the conditions which will cause an interrupt
	from the 8250

	arguments:  ptr to comm parameters, byte flags indicating whether or not
		modem status, line status, transmitter empty, or data ready
		interrupts are to occur
	returns:  nothing
	resources used:  none
	comments:  0 = that interrupt disabled, non-zero = that interrupt enabled
****************************************************************************/
void _8250_setinterrupts (IOPARMS *parms, BOOLEAN modemstatus,
	BOOLEAN linestatus, BOOLEAN xmtrempty, BOOLEAN dataready) {
	UBYTE temp;

	temp = modemstatus ? 0x08 : 0;
	temp |= linestatus ? 0x04 : 0;
	temp |= xmtrempty ? 0x02 : 0;
	temp |= dataready ? 0x01: 0;
	disable ();
	outportb (_IER, temp);						/* configure UART */
	enable ();
}


/****************************************************************************
_8250_getlinestatus():  returns the 8250 line status

	arguments:  ptr to comm parameters
	returns:  the comm port line status flags
	resources used:  the comm port
	comments:  use caution using this with an interrupt driven comm routine.
		Reading the LSR clears several bits.  If the ISR reads the LSR, the
		reported line condition may not be correct, since the bits will have
		been reset by the ISR.
****************************************************************************/
UBYTE _8250_getlinestatus (IOPARMS *parms) {
	return (inportb (_LSR));
}


/****************************************************************************
_8250_getmodemstatus():  returns the status of the 8250 modem control lines

	arguments:  ptr to comm parameters
	returns:  the modem status flags
	resources used:  the comm port
	comments:  use caution using this with an interrupt driven comm routine.
		Reading the MSR clears several bits.  If the ISR reads the MSR, the
		reported conditions may not be correct, since the bits will have
		been reset by the ISR.
****************************************************************************/
UBYTE _8250_getmodemstatus (IOPARMS *parms) {
	return (inportb (_MSR));
}


/****************************************************************************
_8250_RTSon():  activates the request to send line of the 8250

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  used to activate the RS-485 driver on COM2 (if configured for
		RS-485 operation)
****************************************************************************/
void _8250_RTSon (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) | 0x02);
	enable();
}


/****************************************************************************
_8250_RTSoff():  deactivates the request to send line of 8250

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  used to deactivate the RS-485 driver on COM2 (if configured for
		RS-485 operation)
****************************************************************************/
void _8250_RTSoff (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) & ~0x02);
	enable();
}


/****************************************************************************
_8250_DTRon():  enables the comm port DTR output

	arguments:  comm parameter struct for the port
	returns:  nothing
	resources used:  the comm port
	comments:  COM1 and COM2 on the Chickadee do not use the 8250 DTR
		line for serial communications.  This function is included for the
		sake of completeness.  If you have 8250 ports on your I/O board, you
		may need this function.
****************************************************************************/
void _8250_DTRon (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) | 0x01);
	enable();
}


/****************************************************************************
_8250_DTRoff():  disables the 8250 DTR output

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  COM1 and COM2 on the Chickadee do not use the 8250 DTR
		line for serial communications.  This function is included for the
		sake of completeness.  If you have 8250 ports on your I/O board, you
		may need this function.
****************************************************************************/
void _8250_DTRoff (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) & ~0x01);
	enable();
}


/****************************************************************************
_8250_loopenable():  enables 8250 local loopback test mode

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  typically used only for testing
****************************************************************************/
void _8250_loopenable (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) | 0x10);
	enable();
}


/****************************************************************************
_8250_loopdisable():  disables 8250 local loopback test mode

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  typically used only for testing
****************************************************************************/
void _8250_loopdisable (IOPARMS *parms) {
	disable();
	outportb (_MCR, inportb (_MCR) & ~0x10);
	enable();
}


/****************************************************************************
_8250_setbreak():  activates the 8250 break condition

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:  a break forces the serial output line low until the break is
		cleared
****************************************************************************/
void _8250_setbreak (IOPARMS *parms) {
	disable();
	outportb (_LCR, inportb (_LCR) | 0x40);		/* enable break */
	enable();
}


/****************************************************************************
_8250_clearbreak():  deactivates the 8250 break condition

	arguments:  ptr to comm parameters
	returns:  nothing
	resources used:  the comm port
	comments:
****************************************************************************/
void _8250_clearbreak (IOPARMS *parms) {
	disable();
	outportb (_LCR, inportb (_LCR) & ~0x40);		/* disable break */
	enable();
}


/****************************************************************************
_8250_init():  initializes an 8250 for serial communications by setting the
	baud rate, data bits, parity, stop bits, and interrupt conditions

	arguments:	1) ptr to comm parameters
				2) baud divisor
				3) # of data bits (5-8)
				4) parity type (0-2) 0 = none, 1 = odd, 2 = even
				5) # of stop bits (1 or 2)
				6) modem status interrupt (0 = disable, 1 = enable)
				7) receiver line status interrupt (0 = disable, 1 = enable)
				8) transmitter holding register empty interrupt (0 = disable,
					1 = enable)
				9) received data available interrupt (0 = disable,
					1 = enable)
	returns:  nothing
	resources used:  the comm port
	comments:  DOES NOT UNMASK THE CPU INTERRUPT OR ENABLE THE 8250 INTERRUPT
		PIN
****************************************************************************/
void _8250_init (IOPARMS *parms, UWORD bauddiv, UBYTE databits,
	UBYTE parity, UBYTE stopbits, BOOLEAN modemstatus, BOOLEAN linestatus,
	BOOLEAN xmtrempty, BOOLEAN dataready) {

	_8250_setbaud (parms, bauddiv);
	_8250_setdatabits (parms, databits);
	_8250_setparity (parms, parity);
	_8250_setstopbits (parms, stopbits);
	_8250_setinterrupts (parms, modemstatus, linestatus, xmtrempty,
		dataready);

	/* flush any pending interrupts */
	inportb (_LSR);
	inportb (_RBR);
	inportb (_IIR);
	inportb (_MSR);
}


/****************************************************************************
_8250_rcvrready():  checks if a char has been received by 8250

	arguments:  ptr to comm parameters
	returns:  0 if no data is present, non-zero if data is present
	resources used:  the comm port
	comments:  use caution using this with an interrupt driven comm routine.
		Reading the LSR clears several bits.  If the ISR reads the LSR, the
		reported line condition may not be correct, since the bits will have
		been reset by the ISR.
****************************************************************************/
BOOLEAN _8250_rcvrready (IOPARMS *parms) {
	return (inportb (_LSR) & 0x01);
}


/****************************************************************************
_8250_xmtrready():  checks if 8250 is ready to send a char

	arguments:  ptr to comm parameters
	returns:  0 if xmtr is not ready, non-zero if xmtr is ready
	resources used:  the comm port
	comments:  use caution using this with an interrupt driven comm routine.
		Reading the LSR clears several bits.  If the ISR reads the LSR, the
		reported line condition may not be correct, since the bits will have
		been reset by the ISR.
****************************************************************************/
BOOLEAN _8250_xmtrready (IOPARMS *parms) {
	return (inportb (_LSR) & 0x20);		/* xmtr holding register empty flag */
}


/****************************************************************************
_8250_xmtrdone():  checks if 8250 is finished sending a char

	arguments:  ptr to comm parameters
	returns:  0 if xmtr is busy, non-zero if xmtr is done
	resources used:  the comm port
	comments:  use caution using this with an interrupt driven comm routine.
		Reading the LSR clears several bits.  If the ISR reads the LSR, the
		reported line condition may not be correct, since the bits will have
		been reset by the ISR.
****************************************************************************/
BOOLEAN _8250_xmtrdone (IOPARMS *parms) {
	return (inportb (_LSR) & 0x40);		/* xmtr empty flag */
}


/****************************************************************************
_8250_putc():  send char via 8250

	arguments:  ptr to comm parameters, the char to be sent
	returns:  the char to be sent
	resources used:  the comm port
	comments:  assumes the transmitter holding register is available; check
		with _8250_xmtrready()
****************************************************************************/
BYTE _8250_putc (IOPARMS *parms, BYTE c) {
	outportb (_THR, c);
	return (c);
}


/****************************************************************************
_8250_getc():  get char from 8250

	arguments:  ptr to comm parameters
	returns:  char that was received
	resources used:  the comm port
	comments:  assumes there is a char available; check with _8250_rcvrready()
****************************************************************************/
BYTE _8250_getc (IOPARMS *parms) {
	return (inportb (_RBR));
}
